<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Inertia\Inertia;
use Auth;

use App\Models\App;
use App\Models\Setting;
use App\Models\SocialType;
use Illuminate\Validation\Rule;
use Exception;




class AppController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $apps = App::where('status', true)->orderBy('id', 'desc')->get();
        $social_types = SocialType::where('status', true)->get();

        $references = [
            'reference1' => Setting::first()?->reference1,
            'reference2' => Setting::first()?->reference2,
        ];

        $messages = [
            'errors' => $request->session()->get('errors'),
            'message' => $request->session()->get('message'),
            'success' => $request->session()->get('success'),
        ];

        return Inertia::render('Apps/Index', ['apps' => $apps, 'social_types' => $social_types, 'messages' => $messages, 'references' => $references]);
    }

    public function Profiles(Request $request)
    {

        $user = Auth::user();

        $profiles = [];

        if (!empty($user->managed_profiles) && $user->managed_profiles != '') {
            $profiles = App::where('status', true)->where("published", true)->whereIn('id', explode(',', $user->managed_profiles))->get();
        }


        $messages = [
            'errors' => $request->session()->get('errors'),
            'message' => $request->session()->get('message'),
            'success' => $request->session()->get('success'),
        ];

        return Inertia::render('Profiles/Index', ['profiles' => $profiles, 'messages' => $messages]);
    }

    public function checkCode(Request $request)
    {
        $code = strtolower($request->query('code'));
        $id = $request->query('id');

        $query = App::where('code', $code)->where('status', true);

        if ($id) {
            $query->where('id', '!=', $id);
        }

        $app = $query->first();

        if ($app) {
            return response()->json(['status' => 200, 'message' => 'Code already used!'], 200);
        }

        return response()->json(['status' => 404, 'message' => 'Code available!'], 404);
    }

    public function store(Request $request)
    {
        try {
            $validated = $request->validate([
                'name' => 'string|nullable',
                'code' => [
                    'string',
                    'max:50',
                    'nullable',
                ],
                'number' => 'string|max:50|nullable',
                'mobile' => 'string|max:50|nullable',
                'subscription_date' => 'string|max:50|nullable',
                'subscription_end_date' => 'string|max:50|nullable',
                'description' => 'string|nullable',
                'published' => 'boolean|nullable',
                'address' => 'string|nullable',
                'template' => 'string|max:50|required',
                'powered_by' => 'string|max:50|nullable',
                'gender' => 'string|max:50|nullable',
                'reference1' => 'string|nullable',
                'reference2' => 'string|nullable',
                'services_section' => 'string|nullable',
                'blocks_section' => 'string|nullable',
                'embeds_section' => 'string|nullable',
                'testimonials_section' => 'string|nullable',
                'blocks' => 'array|nullable',
                'services' => 'array|nullable',
                'testimonials' => 'array|nullable',
            ]);



            if ($validated['code'] && App::where('code', strtolower($validated['code']))->where('status', true)->exists()) {
               
                session()->flash('errors', 'Code already used!');

                return Inertia::location('/admin/contacts');
            }


            if ($validated['published'] == true) {
                if ($validated['code'] == '' || !isset($validated['code'])) {
                     
                    session()->flash('errors', 'Code is required!');
                    return Inertia::location('/admin/contacts');
                }
            }



            $app = new App();

            $app->name = isset($validated['name']) ? str_replace('&nbsp;', '', $validated['name']) : '';
            $app->code = strtolower($validated['code']) ?? '';
            $app->number = $validated['number'] ?? '';
            $app->mobile = $validated['mobile'] ?? '';
            $app->description = isset($validated['description']) ? str_replace('&nbsp;', '', $validated['description']) : '';
            $app->address = isset($validated['address']) ? str_replace('&nbsp;', '', $validated['address']) : '';
            $app->template = $validated['template'] ?? '';
            $app->published = $validated['published'] ?? false;
            $app->powered_by = $validated['powered_by'] ?? '';
            $app->gender = $validated['gender'] ?? '';
            $app->subscription_date = $validated['subscription_date'] ?? null;
            $app->subscription_end_date = $validated['subscription_end_date'] ?? null;
            $app->reference1 = $validated['reference1'] ?? '';
            $app->reference2 = $validated['reference2'] ?? '';
            $app->blocks_section = $validated['blocks_section'] ?? '';
            $app->embeds_section = $validated['embeds_section'] ?? '';
            $app->services_section = $validated['services_section'] ?? '';
            $app->testimonials_section = $validated['testimonials_section'] ?? '';
            $app->created_by = Auth::user()->name;

            if ($request->file('profile_pic')) {
                // Handle file upload
                $fileName = time() . '_' . $request->file('profile_pic')->getClientOriginalName(); // Generate a unique file name
                $filePath = $request->file('profile_pic')->storeAs('assets', $fileName, 'public'); // Store the file in the 'public/assets' directory

                if (env('APP_ENV') == 'local') {
                    $app->profile_pic = "/storage/assets/$fileName";
                } else {
                    $app->profile_pic = "/storage/app/public/assets/$fileName";
                }
            } else {
                $app->profile_pic = $request->get('profile_pic') ?? '';
            }

            if ($request->file('cover_pic')) {
                // Handle file upload
                $fileName = time() . '_' . $request->file('cover_pic')->getClientOriginalName(); // Generate a unique file name
                $filePath = $request->file('cover_pic')->storeAs('assets', $fileName, 'public');// Store the file in the 'public/assets' directory

                if (env('APP_ENV') == 'local') {
                    $app->cover_pic = "storage/assets/$fileName";
                } else {
                    $app->cover_pic = "/storage/app/public/assets/$fileName";
                }

            } else {
                $app->cover_pic = $request->get('cover_pic') ?? '';
            }




            $app->save();


            if (is_array($request->get("social_links"))) {
                foreach ($request->get("social_links") as $link) {

                    $data = [
                        "social_type_id" => $link["social_type_id"] ?? '',
                        "link" => $link["link"] ?? '',
                        "order" => $link['order'] ?? 1
                    ];


                    $app->social_links()->create($data);
                }
            }

            if (is_array($request->get("embeds"))) {
                foreach ($request->get("embeds") as $embed) {

                    $data = [
                        "embed" => $embed["embed"] ?? '',
                        "order" => $embed['order'] ?? 1
                    ];

                    $app->embeds()->create($data);
                }
            }

             if (is_array($request->get("blocks"))) {
             foreach($validated['blocks'] as $block) {
              
                $app_block = [
                        'title' => $block['title'] ?? '',
                        'description' => $block['description'] ?? '',
                        'background' => $block['background'] ?? '',
                        'link' => $block['link'] ?? '',
                        'button' => $block['button'] ?? '',
                        'order' => $block['order'] ?? 1,
                        'app_id' => $app->id,
                ];

                if (isset($block['image']) && $block['image'] instanceof \Illuminate\Http\UploadedFile) {
                    // Handle image upload
                    $fileName = time() . '_' . $block['image']->getClientOriginalName(); // Generate a unique file name
                    $filePath = $block['image']->storeAs('assets', $fileName, 'public'); // Store the file in the 'public/assets' directory

                    if (env('APP_ENV') == 'local') {
                        $app_block['image'] = "/storage/assets/$fileName";
                    } else {
                        $app_block['image'] = "/storage/app/public/assets/$fileName";
                    }
                }else{
                    $app_block['image'] = "";
                }

                    
                    $app->blocks()->create($app_block);
            }
        }


        if (is_array($request->get("services"))) {
            foreach($validated['services'] as $service) {
              
                $app_service = [
                        'title' => $service['title'] ?? '',
                        'description' => $service['description'] ?? '',
                        'background' => $service['background'] ?? '',
                        'link' => $service['link'] ?? '',
                        'button' => $service['button'] ?? '',
                        'order' => $service['order'] ?? 1,
                        'app_id' => $app->id,
                ];

                if (isset($service['image']) && $service['image'] instanceof \Illuminate\Http\UploadedFile) {
                    // Handle image upload
                    $fileName = time() . '_' . $service['image']->getClientOriginalName(); // Generate a unique file name
                    $filePath = $service['image']->storeAs('assets', $fileName, 'public'); // Store the file in the 'public/assets' directory

                    if (env('APP_ENV') == 'local') {
                        $app_service['image'] = "/storage/assets/$fileName";
                    } else {
                        $app_service['image'] = "/storage/app/public/assets/$fileName";
                    }
                }else{
                    $app_service['image'] = "";
                }

                    
                    $app->services()->create($app_service);
            }
        }


        if (is_array($request->get("testimonials"))) {
            foreach ($validated['testimonials'] as $testimonial) {

                $app_testimonial = [
                    'name' => $testimonial['name'] ?? '',
                    'description' => $testimonial['description'] ?? '',
                    'order' => $testimonial['order'] ?? 1,
                    'app_id' => $app->id,
                ];

                if (isset($testimonial['image']) && $testimonial['image'] instanceof \Illuminate\Http\UploadedFile) {
                    // Handle image upload
                    $fileName = time() . '_' . $testimonial['image']->getClientOriginalName(); // Generate a unique file name
                    $filePath = $testimonial['image']->storeAs('assets', $fileName, 'public'); // Store the file in the 'public/assets' directory

                    if (env('APP_ENV') == 'local') {
                        $app_testimonial['image'] = "/storage/assets/$fileName";
                    } else {
                        $app_testimonial['image'] = "/storage/app/public/assets/$fileName";
                    }
                } else {
                    $app_testimonial['image'] = "";
                }

                $app->testimonials()->create($app_testimonial);
            }
        }
            

            session()->flash('success', 'Contact created succesfully!');
            return redirect()->route('contacts.index');
        } catch (Exception $e) {
            return response()->json(['message' => $e->getMessage(), 'status' => 500], 500);
        }

    }


    public function update(Request $request, string $id)
    {
        try {

            $validated = $request->validate([
                'name' => 'string|nullable',
                'code' => [
                    'string',
                    'max:50',
                    'nullable',
                ],
                'number' => 'string|max:50|nullable',
                'mobile' => 'string|max:50|nullable',
                'description' => 'string|nullable',
                'address' => 'string|nullable',
                'subscription_date' => 'string|max:50|nullable',
                'subscription_end_date' => 'string|max:50|nullable',
                'template' => 'string|max:50',
                'published' => 'boolean|nullable',
                'powered_by' => 'string|max:50|nullable',
                'gender' => 'string|max:50|nullable',
                'reference1' => 'string|nullable',
                'reference2' => 'string|nullable',
                'blocks_section' => 'string|nullable',
                'embeds_section' => 'string|nullable',
                'services_section' => 'string|nullable',
                'testimonials_section' => 'string|nullable',
                'blocks' => 'array|nullable',
                'services' => 'array|nullable',
                'testimonials' => 'array|nullable'
            ]);



            if ($validated['code'] && App::where('code', strtolower($validated['code']))->where('status', true)->where('id', '!=', $id)->exists()) {
                session()->flash('errors', 'Code already used!');
                return Inertia::location('/admin/contacts');
            }

            if ($validated['published'] == true) {
                if ($validated['code'] == '' || !isset($validated['code'])) {
                    session()->flash('errors', 'Code is required!');
                    return Inertia::location('/admin/contacts');
                }
            }

            $app = App::with("social_links")->with("embeds")->with('blocks')->with('services')->with('testimonials')->find($id);

            $app->name = isset($validated['name']) ? str_replace('&nbsp;', '', $validated['name']) : $app->name;
            $app->code = strtolower($validated['code']) ?? '';
            $app->number = $validated['number'] ?? $app->number;
            $app->mobile = $validated['mobile'] ?? $app->mobile;
            $app->description = isset($validated['description']) ? str_replace('&nbsp;', '', $validated['description']) : $app->description;
            $app->address = isset($validated['address']) ? str_replace('&nbsp;', '', $validated['address']) : $app->address;
            $app->template = $validated['template'] ?? $app->template;
            $app->published = $validated['published'] ?? $app->published;
            $app->powered_by = $validated['powered_by'] ?? $app->powered_by;
            $app->gender = $validated['gender'] ?? $app->gender;
            $app->subscription_date = $validated['subscription_date'] ?? $app->subscription_date;
            $app->subscription_end_date = $validated['subscription_end_date'] ?? $app->subscription_end_date;
            $app->reference1 = $validated['reference1'] ?? $app->reference1;
            $app->reference2 = $validated['reference2'] ?? $app->reference2;
            $app->blocks_section = $validated['blocks_section'] ?? '';
            $app->embeds_section = $validated['embeds_section'] ?? '';
            $app->services_section = $validated['services_section'] ?? '';
            $app->testimonials_section = $validated['testimonials_section'] ?? '';
            $app->updated_by = Auth::user()->name;

            if ($request->file('profile_pic')) {
                // Handle file upload
                $fileName = time() . '_' . $request->file('profile_pic')->getClientOriginalName(); // Generate a unique file name
                $filePath = $request->file('profile_pic')->storeAs('assets', $fileName, 'public'); // Store the file in the 'public/assets' directory

                if (env('APP_ENV') == 'local') {
                    $app->profile_pic = "storage/assets/$fileName";
                } else {
                    $app->profile_pic = "/storage/app/public/assets/$fileName";
                }

            } else {
                $app->profile_pic = $request->get('profile_pic') ?? '';
            }

            if ($request->file('cover_pic')) {
                // Handle file upload
                $fileName = time() . '_' . $request->file('cover_pic')->getClientOriginalName(); // Generate a unique file name
                $filePath = $request->file('cover_pic')->storeAs('assets', $fileName, 'public');// Store the file in the 'public/assets' directory

                if (env('APP_ENV') == 'local') {
                    $app->cover_pic = "storage/assets/$fileName";
                } else {
                    $app->cover_pic = "/storage/app/public/assets/$fileName";
                }
            } else {
                $app->cover_pic = $request->get('cover_pic') ?? '';
            }

            $app->save();

            foreach ($app->social_links as $link) {
                $link->delete();
            }

            if (is_array($request->get("social_links"))) {
                foreach ($request->get("social_links") as $link) {

                    $data = [
                        "social_type_id" => $link["social_type_id"] ?? '',
                        "link" => $link["link"] ?? '',
                        "order" => $link['order'] ?? 1
                    ];

                    $appId = (int) $app['id'];

                    $app->social_links()->create($data);
                }
            }


            foreach ($app->embeds as $embed) {
                $embed->delete();
            }



            if (is_array($request->get("embeds"))) {
                foreach ($request->get("embeds") as $embed) {

                    $data = [
                        "embed" => $embed["embed"] ?? '',
                        "order" => $embed['order'] ?? 1
                    ];

                    $app->embeds()->create($data);
                }
            }


            foreach ($app->blocks as $block) {
                $block->delete();
            }

            foreach ($validated['blocks'] as $block) {

                $app_block = [
                    'title' => $block['title'] ?? '',
                    'description' => $block['description'] ?? '',
                    'background' => $block['background'] ?? '',
                    'button' => $block['button'] ?? '',
                    'link' => $block['link'] ?? '',
                    'order' => $block['order'] ?? 1,
                    'app_id' => $app->id,
                ];

                if (isset($block['image']) && $block['image'] instanceof \Illuminate\Http\UploadedFile) {
                    // Handle image upload
                    $fileName = time() . '_' . $block['image']->getClientOriginalName(); // Generate a unique file name
                    $filePath = $block['image']->storeAs('assets', $fileName, 'public'); // Store the file in the 'public/assets' directory

                    if (env('APP_ENV') == 'local') {
                        $app_block['image'] = "/storage/assets/$fileName";
                    } else {
                        $app_block['image'] = "/storage/app/public/assets/$fileName";
                    }
                } else{
                    $app_block['image'] = $block['image'] ?? "";
                }

                $app->blocks()->create($app_block);
            }


            foreach ($app->services as $service) {
                $service->delete();
            }

            foreach ($validated['services'] as $service) {

                $app_service = [
                    'title' => $service['title'] ?? '',
                    'description' => $service['description'] ?? '',
                    'background' => $service['background'] ?? '',
                    'button' => $service['button'] ?? '',
                    'link' => $service['link'] ?? '',
                    'order' => $service['order'] ?? 1,
                    'app_id' => $app->id,
                ];

                if (isset($service['image']) && $service['image'] instanceof \Illuminate\Http\UploadedFile) {
                    // Handle image upload
                    $fileName = time() . '_' . $service['image']->getClientOriginalName(); // Generate a unique file name
                    $filePath = $service['image']->storeAs('assets', $fileName, 'public'); // Store the file in the 'public/assets' directory

                    if (env('APP_ENV') == 'local') {
                        $app_service['image'] = "/storage/assets/$fileName";
                    } else {
                        $app_service['image'] = "/storage/app/public/assets/$fileName";
                    }
                } else {
                    $app_service['image'] = $service['image'];
                }

                $app->services()->create($app_service);
            }



            foreach ($app->testimonials as $testimonial) {
                $testimonial->delete();
            }

            foreach ($validated['testimonials'] as $testimonial) {

                $app_testimonial = [
                    'name' => $testimonial['name'] ?? '',
                    'description' => $testimonial['description'] ?? '',
                    'order' => $testimonial['order'] ?? 1,
                    'app_id' => $app->id,
                    
                ];

                if (isset($testimonial['image']) && $testimonial['image'] instanceof \Illuminate\Http\UploadedFile) {
                    // Handle image upload
                    $fileName = time() . '_' . $testimonial['image']->getClientOriginalName(); // Generate a unique file name
                    $filePath = $testimonial['image']->storeAs('assets', $fileName, 'public'); // Store the file in the 'public/assets' directory

                    if (env('APP_ENV') == 'local') {
                        $app_testimonial['image'] = "/storage/assets/$fileName";
                    } else {
                        $app_testimonial['image'] = "/storage/app/public/assets/$fileName";
                    }
                } else{
                    $app_testimonial['image'] = $testimonial['image'];
                }

                $app->testimonials()->create($app_testimonial);
            }



            $user = Auth::user()->with('roles')->find(Auth::user()->id);

            if ($user->roles->contactsF == 1) {
                session()->flash('success', 'Contact updated successfully!');
                return Inertia::location('/admin/contacts');
            }

            session()->flash('success', 'Profile updated successfully!');
            return Inertia::location('/admin/profiles');
        } catch (\Exception $e) {
            return response()->json(['message' => $e->getMessage(), 'status' => 500], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try {
            $app = App::find($id);
            $app->status = false;

            if ($app->save()) {
                session()->flash('success', 'Contact deleted successfully!');
                return Inertia::location('/admin/contacts');
            }

            session()->flash('errors', 'Could not delete contact!');
            return redirect()->back();
        } catch (Exception $e) {
            return response()->json(['message' => $e->getMessage(), 'status' => 500], 500);

        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id, Request $request)
    {
        $selected_contact = App::with('social_links')->with('embeds')->with('blocks')->with('services')->with('testimonials')->where('status', true)->findOrFail($id);
        $social_types = SocialType::where('status', true)->get();
        $apps = App::where('status', true)->orderBy('id', 'desc')->get();

        $messages = [
            'errors' => $request->session()->get('errors'),
            'message' => $request->session()->get('message'),
            'success' => $request->session()->get('success'),
        ];

        $references = [
            'reference1' => Setting::first()?->reference1,
            'reference2' => Setting::first()?->reference2,
        ];

        return Inertia::render('Apps/Update', ['social_types' => $social_types, 'selected_contact' => $selected_contact, 'apps' => $apps, 'messages' => $messages, 'references' => $references]);

    }

    public function EditProfile(string $id, Request $request)
    {
        $selected_contact = App::with('social_links')->with('embeds')->where('status', true)->findOrFail($id);

        $user = Auth::user();

        $managed_profiles = $user->managed_profiles;
        $managed_profiles = explode(',', $managed_profiles);

        if (!in_array($id, $managed_profiles)) {
            session()->flash('errors', 'Unauthorized access! you are not allowed to edit this profile.');
            return redirect()->back();
        }

        $social_types = SocialType::where('status', true)->get();

        $messages = [
            'errors' => $request->session()->get('errors'),
            'message' => $request->session()->get('message'),
            'success' => $request->session()->get('success'),
        ];


        return Inertia::render('Profiles/Update', ['social_types' => $social_types, 'selected_contact' => $selected_contact, 'messages' => $messages]);

    }

    public function duplicate(string $id, Request $request)
    {
        $selected_contact = App::with('social_links')->with('embeds')->with('blocks')->with('services')->with('testimonials')->where('status', true)->findOrFail($id);
        $selected_contact->code = null;
        $social_types = SocialType::where('status', true)->get();
        $apps = App::where('status', true)->orderBy('id', 'desc')->get();

        $messages = [
            'errors' => $request->session()->get('errors'),
            'message' => $request->session()->get('message'),
            'success' => $request->session()->get('success'),
        ];

        $references = [
            'reference1' => Setting::first()?->reference1,
            'reference2' => Setting::first()?->reference2,
        ];

        return Inertia::render('Apps/Duplicate', ['social_types' => $social_types, 'selected_contact' => $selected_contact, 'apps' => $apps, 'messages' => $messages, 'references' => $references]);

    }

    public function togglePublish(string $id)
    {
        $app = App::findOrFail($id);


        $app->published = !$app->published;

        if ($app->published && !$app->code) {
            session()->flash('errors', 'Code is required!');
            return Inertia::location('/admin/contacts');
        }

        if ($app->save()) {
            session()->flash('success', 'Contact updated successfully!');

            return Inertia::location('/admin/contacts');
        }

        session()->flash('errors', 'Could not update contact!');
        return redirect()->back();
    }


    public function showApp(Request $request,string $code)
    {
        
        $App = App::with("social_links.social_type")
            ->with('embeds')
            ->with('blocks')
            ->with('services')
            ->with('testimonials')
            ->where('code', strtolower($code))
            ->where('published', true)
            ->where('status', true)
            ->firstOrFail();

        $user = Auth::user();

        if (!empty($user->managed_profiles) && $user->managed_profiles != '') {
            $managed_profiles = explode(',', $user->managed_profiles);
            if (in_array($App->id, $managed_profiles)) {
                //can edit profile
                $App->can_edit = true;
            }
        }

        if ($App->subscription_end_date && $App->subscription_end_date < now()) {
            session()->flash('errors', 'Subscription expired! Please contact the admin.');

            return Abort(403, 'Please contact the admin.');
        }



        if (!is_null($App)) {
            $template = $request->query('template') ?? $App->template;
            
            if (is_null($template)) {
                $template = 'Template1';
            }

            $template = ucfirst(strtolower($template));

            return Inertia::render('Apps/templates/' . $template, ['app' => $App]);

        }

        return 'Not Found';

    }

    public function generateVCard(string $code)
    {
        try {
            $App = App::where('code', $code)->where('status', 1)->first();

            if (!is_null($App)) {
                $content = view('vcard', compact(['App']))->render();

                return response($content, 200)
                    ->header('Content-Type', 'text/vcard')
                    ->header('Content-Disposition', 'attachment; filename="' . strip_tags($App->name) . '.vcf"');
            }

            return 'Not Found';
        } catch (Exception $e) {
            return response()->json(['message' => $e->getMessage(), 'status' => 500], 500);
        }

    }
}
