<?php

namespace App\Http\Controllers;

use App\Models\Language;
use Exception;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class LanguageController extends Controller implements HasMiddleware
{
    public static function middleware(): array
    {
        return [
            new Middleware('CheckPermission:languagesA', only: ['store']),
            new Middleware('CheckPermission:languagesE', only: ['update', 'edit']),
            new Middleware('CheckPermission:languagesD', only: ['destroy'])
        ];
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $languages = Language::all();

            $messages = [
                'errors' => $request->session()->get('errors'),
                'message' => $request->session()->get('message'),
                'success' => $request->session()->get('success'),
            ];


            return Inertia::render('Languages/Index')->with([
                'languages' => $languages,
                'messages' => $messages,
            ]);


        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred please try again later.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try{
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'flag' => 'nullable|string',
                'locale' => 'required|string|max:10',
                'status' => 'required|boolean',
            ]);

            // Check if the language already exists
            if (Language::where('locale', $validated['locale'])->orWhere('name', $validated['name'])->exists()) {
                return response()->json([
                    'error' => 'Language with this locale already exists.'
                ], 422);
            }

            $language = new Language();
            $language->name = $validated['name'];
            $language->flag = $validated['flag'];
            $language->locale = $validated['locale'];
            $language->status = $validated['status'];
            $language->save();

            // Flash success message to the session
            session()->flash('success', 'Language created successfully!');
            return Inertia::location(route('languages.index'));

        }catch(Exception $e){
            return response()->json([
                'error' => 'An error occurred while creating the language.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Request $request , string $id)
    {
        try {
            $languages = Language::all();
            $selected_language = Language::findOrFail($id);

            $messages = [
                'errors' => $request->session()->get('errors'),
                'message' => $request->session()->get('message'),
                'success' => $request->session()->get('success'),
            ];


            return Inertia::render('Languages/Update')->with([
                'languages' => $languages,
                'selected_language' => $selected_language,
                'messages' => $messages,
            ]);


        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred please try again later.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'flag' => 'nullable|string',
                'locale' => 'required|string|max:10',
                'status' => 'required|boolean',
                'is_rtl' => 'nullable|boolean',
            ]);

            $language = Language::findOrFail($id);
            
            // Check if the language already exists with a different ID
            if (Language::where('locale', $validated['locale'])
                ->where('id', '!=', $language->id)
                ->orWhere('name', $validated['name'])
                ->where('id', '!=', $language->id)
                ->exists()) {
                return response()->json([
                    'error' => 'Language with this locale or name already exists.'
                ], 422);
            }

            $language->name = $validated['name'];
            $language->flag = $validated['flag'];
            $language->locale = $validated['locale'];
            $language->status = $validated['status'];
            $language->is_rtl = $validated['is_rtl'] ?? false;
            $language->save();

            // Flash success message to the session
            session()->flash('success', 'Language updated successfully!');
            return Inertia::location(route('languages.index'));

        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred while updating the language.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        try{
            $language = Language::findOrFail($id);
            $language->delete();

            // Flash success message to the session
            session()->flash('success', 'Language deleted successfully!');
            return Inertia::location(route('languages.index'));

        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred while deleting the language.',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
