import type { Editor, ObjectAny } from 'grapesjs';
import { ReactNode } from 'react';
import { StudioButtonProps } from '../components/public/StudioButton';
import { IconNames } from '../components/public/StudioIcon';
export interface WithEditorProps {
    editor: Editor;
}
export interface CategoryBaseProps {
    id: string;
    label?: string;
    open?: boolean;
}
export interface WithCategoryBase {
    category?: CategoryBaseProps;
}
export interface ItemsByCategory<T> {
    category?: CategoryBaseProps;
    items: T[];
}
export interface ItemWithCategory<T> {
    category?: CategoryBaseProps;
    item: T;
}
export interface ItemsWithCategory<T> extends ItemsByCategory<T> {
    category: CategoryBaseProps;
}
export type CustomItems<T> = T[] | ((props: {
    items: T[];
} & WithEditorProps) => T[]) | false;
export interface CommandItem<T extends ObjectAny = ObjectAny> extends Pick<StudioButtonProps, 'label'> {
    /**
     * Id of the command
     */
    id: string;
    /**
     * Command to trigger on click.
     * cmd: ({ editor }) => {
     *   alert('Current HTML:', editor.getHtml())
     * },
     */
    cmd?: (props: WithEditorProps & T) => void;
}
export interface CommandButtonItem<T extends ObjectAny = ObjectAny> extends Omit<CommandItem<T>, 'label' | 'cmd'>, StudioButtonProps {
    /**
     * Id of the action
     */
    id: string;
}
export interface ActionButton {
    /**
     * Id of the action
     */
    id: string;
    /**
     * Command to trigger on click.
     * You can pass a string of defined GrapesJS [commands](https://grapesjs.com/docs/modules/Commands.html) or a custom logic as a function.
     * @example
     * cmd: 'core:undo',
     * // or
     * cmd: ({ editor }) => {
     *   alert('Current HTML:', editor.getHtml())
     * },
     */
    cmd: string | ((props: WithEditorProps) => void);
    /**
     * Content to use inside the button.
     * If the string starts with `<svg`, it will be rendered as an SVG.
     */
    content?: ReactNode;
    /**
     * Title of the action that will be displayed on hover in a tooltip.
     */
    title: string;
    /**
     * Indicates if the button should be disabled.
     * @example
     * isDisabled: ({ editor }) => {
     *  // Disable the button if no component is selected
     *  return !editor.getSelected();
     * }
     */
    isDisabled?: (props: WithEditorProps) => boolean;
    iconPath?: string;
    iconName?: `${IconNames}` | IconNames;
    options?: Record<string, any>;
}
export declare enum BuiltInActionButtons {
    componentOutline = "componentOutline",
    preview = "preview",
    fullscreen = "fullscreen",
    showCode = "showCode",
    showImportCode = "showImportCode",
    clearCanvas = "clearCanvas",
    store = "store",
    undo = "undo",
    redo = "redo"
}
export interface CommandDefaultOptions {
    [BuiltInActionButtons.showCode]: CommandShowCodeOptions;
}
export interface CommandShowCodeOptions {
    /**
     * Customize the export option.
     * @example
     * // Hide the export button.
     * export: false
     *
     */
    export?: boolean;
}
