import { ProjectData } from '../utils/types';
import { WithEditorProps } from './common';
export declare enum ProjectStorageType {
    browser = "browser",
    self = "self",
    cloud = "cloud"
}
export interface ProjectDataResult {
    id?: string;
    project: ProjectData;
}
export interface StorageConfig {
    /**
     * Choose between saving project data on:
     * - 'browser': the browser
     * - 'self': your own self-hosted infrastructure
     * - 'cloud': our project cloud
     *
     * To use 'cloud' you must provide an identity.id for your users and a project.id for your project in config.
     * @default 'browser'
     */
    type?: `${ProjectStorageType}` | ProjectStorageType;
    /**
     * Provide a custom handler for saving the project data.
     * If the handler doesn't throw any error, the project will be considered properly saved.
     * @example
     * onSave: async ({ project }) => {
     *   const body = new FormData();
     *   body.append('project', project);
     *   const response = await fetch('PROJECT_SAVE_URL', { method: 'POST', body });
     * }
     */
    onSave?: (props: {
        project: ProjectData;
    } & WithEditorProps) => Promise<any>;
    /**
     * Provide a custom handler for loading project data.
     * The handler should return an object with the project data.
     * @example
     * onLoad: async () => {
     *   const response = await fetch('PROJECT_LOAD_URL');
     *   const project = await response.json();
     *   return {
     *    project,
     *   }
     * }
     */
    onLoad?: (props: WithEditorProps) => Promise<ProjectDataResult>;
    /**
     * If the project JSON is provided, the onLoad handler will be skipped.
     */
    project?: ProjectData;
    /**
     * Indicate after how many changes the project should be autosaved.
     * Note: with no unsaved changes, the save won't be triggered
     */
    autosaveChanges?: number;
    /**
     * Indicate after how many milliseconds the project should be autosaved.
     * Note: with no unsaved changes, the save won't be triggered.
     */
    autosaveIntervalMs?: number;
}
