import type { Editor } from 'grapesjs';
import { ItemsPerRow } from '../components/public/contentUtils';
import { ProjectItem } from '../utils/types';
export interface StudioPanelTemplatesProps {
    /**
     * Unique id for this layout panel.
     * @example "panel-templates"
     */
    id?: string;
    /**
     * Layout type, must be set to 'panelTemplates'.
     * @example "panelTemplates"
     */
    type: 'panelTemplates';
    /**
     * Custom array of templates to show in this panel.
     * @examples
     * templates: [{
     *   id: 'template1',
     *   name: 'Template 1',
     *   data: {
     *     pages: [
     *       {
     *         name: 'Home',
     *         component: '<h1 class="title">Template 1</h1><style>.title { color: red; font-size: 10rem; text-align: center }</style>'
     *       }
     *     ]
     *   }
     * }]
     */
    templates?: TemplateItem[];
    /**
     * Provide a custom handler for the select button.
     * @examples
     * onSelect: ({ loadTemplate, template }) => {
     *   // loads the selected template to the current project
     *   loadTemplate(template);
     * }
     */
    onSelect?: (props: {
        editor: Editor;
        template: ProjectItem;
        loadTemplate: (template: ProjectItem) => void;
    }) => void;
    /**
     * Extra props to customize this layout panel.
     * @example
     * {
     *   "itemsPerRow": 3
     * }
     */
    content?: {
        itemsPerRow?: ItemsPerRow;
    };
}
export interface TemplateItem {
    /**
     * Unique id for this template item.
     * @example "template1"
     */
    id: string;
    /**
     * Name displayed for this template item.
     * @example "Template 1"
     */
    name: string;
    /**
     * A thumbnail URL for this template.
     * @example "https://example.com/template1.jpg"
     */
    media?: string;
    /**
     * An object containing the name of the author and optionally a link to his socials/website.
     * @example
     * {
     *   name: 'GrapesJS',
     *   link: 'grapesjs.com'
     * }
     */
    author?: {
        name: string;
        link?: string;
    };
    /**
     * GrapesJS project data that will be loaded when the user selects this template.
     * @example
     * {
     *   pages: [
     *     {
     *       name: 'Home',
     *       component: '<h1 class="red-bg">Red background template</h1><style>.red-bg { color: white; background: red; height: 100dvh; }</style>'
     *     }
     *   ]
     * }
     */
    data: Record<string, any>;
}
export interface TemplatesConfig {
    /**
     * Provide a custom handler for loading list of available templates to display in the templates layout panel.
     * It should return an array of TemplateItems.
     * @example
     * onLoad: async ({editor, fetchCommunityTemplates}) => {
     *   const response = await fetch('TEMPLATES_URL');
     *   const templates = await response.json();
     *   return templates;
     * }
     */
    onLoad?: (props: {
        editor: Editor;
        /**
         * @returns a list of templates from the GrapesJS community
         */
        fetchCommunityTemplates: () => Promise<TemplateItem[]>;
    }) => Promise<TemplateItem[]>;
}
