<?php

namespace App\Http\Controllers;

use App\Models\Template;
use Exception;
use Http;
use Illuminate\Http\Request;
use Inertia\Inertia;

use Illuminate\Routing\Controllers\HasMiddleware;
use Illuminate\Routing\Controllers\Middleware;

class TemplateController extends Controller implements HasMiddleware
{

    public static function middleware(): array
    {
        return [
            new Middleware('CheckPermission:templatesA', only: ['store']),
            new Middleware('CheckPermission:templatesE', only: ['update', 'edit', 'quickUpdate']),
            new Middleware('CheckPermission:templatesD', only: ['destroy'])
        ];
    }
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        try {
            $templates = Template::all();

            $messages = [
                'errors' => $request->session()->get('errors'),
                'message' => $request->session()->get('message'),
                'success' => $request->session()->get('success'),
            ];

            return Inertia::render('Templates/Index')->with([
                'templates' => $templates,
                'messages' => $messages,
            ]);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred please try again later.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        try {
            set_time_limit(3600);

            $validated = $request->validate([
                'pages' => 'required|string',
                'prompt' => 'required|string',
            ]);

            $pages = explode(',', $validated['pages']);


            foreach ($pages as $page) {

                /*$templateRequest = Http::timeout(3600)->post('https://hook.eu2.make.com/soqenlbswxweuvg56ognbii6gzwfe1am', [
                    'prompt' => $validated['prompt'],
                    'page' => $page
                ]);*/

                $templateRequest = Http::timeout(3600)->post(env('TEMPLATE_WEBHOOK_URL'), [
                    'template' => $validated['prompt'],
                    'page' => $page
                ]);

                $content = $templateRequest->json()[0]['content'] ?? '';
                $content = str_replace(['```html', '```'], '', $content);

                $page = $validated['prompt'] . ' - ' . $page;

                $template = Template::create([
                    'title' => $page,
                    'component' => $content,
                    'styles' => '',
                ]);
            }
            /*$templateRequest = Http::timeout(3600)->post('https://hook.eu2.make.com/xbj7pve7c5kardaojup5duoae24wwv5b', [
                'prompt' => $validated['prompt'],
                'pages' => $pages
            ]);*/

            /*$templateResult = $templateRequest->json();


            foreach ($templateResult as $result) {
                $parsedResult = json_decode($result['json'], true);

                $template = Template::create([
                    'title' => $parsedResult['page'] ?? 'Untitled Template',
                    'component' => $parsedResult['content'] ?? '',
                    'styles' => '',
                ]);
                $template->save();
            }
*/
            /*$template = Template::create([
                'title' => $validated['title'],
                'component' => $templateRequest->json('value') ?? '',
                'styles' => '',
            ]);*/

            session()->flash('success', 'Templates created successfully!');
            return Inertia::location('/admin/templates/');
        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred while storing the template.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    public function storeApi(Request $request)
    {
        try {

            $validated = $request->validate([
                'page' => 'required|string',
                'component' => 'required|string',
            ]);

            $component = str_replace(['```html', '```'], '', $validated['component']);


            $template = Template::create([
                'title' => $validated['page'],
                'component' => $component,
                'styles' => '',
            ]);

            return response()->json([
                'message' => 'Template created successfully!',
                'template' => $template,
            ], 201);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred while storing the template.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        try {
            $template = Template::findOrFail($id);

            $messages = [
                'errors' => session('errors'),
                'message' => session('message'),
                'success' => session('success'),
            ];

            if (!is_null($id)) {

                if ($template) {
                    $grape_key = env('VITE_GRAPESJS_LICENSE_KEY', null);
                    return Inertia::render('PageBuilder', ['page' => $template, 'redirect' => 'templates', 'grape_key' => $grape_key])->with([
                        'messages' => $messages,
                    ]);
                } else {
                    session()->flash('errors', 'Page not found!');
                    return redirect()->back();
                }
            }

        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred while fetching the template.',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        try {
            $validated = $request->validate([
                'title' => 'required|string|max:255',
                'component' => 'nullable|string',
                'styles' => 'nullable|string',
            ]);

            $template = Template::findOrFail($id);

            $template->title = $validated['title'];
            $template->component = $validated['component'] ?? $template->component;
            $template->styles = $validated['styles'] ?? $template->styles;
            $template->save();

            return response()->json([
                'message' => 'Template updated successfully!',
                'redirect' => route('templates.index')
            ], 200);
        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred while updating the template.',
                'message' => $e->getMessage()
            ], 500);
        }

    }

    public function quickUpdate(Request $request, string $id)
    {
        try {
            $validated = $request->validate([
                'title' => 'required|string|max:255',
            ]);

            $template = Template::findOrFail($id);

            $template->title = $validated['title'];
            $template->save();

            session()->flash('success', 'Template updated successfully!');
            return Inertia::location(route(name: 'templates.index'));
        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred while updating the template.',
                'message' => $e->getMessage()
            ], 500);
        }

    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Template $template)
    {
        try {
            $template->delete();
            session()->flash('success', 'Template deleted successfully!');
            return Inertia::location('/admin/templates');
        } catch (Exception $e) {
            return response()->json([
                'error' => 'An error occurred while deleting the template.',
                'message' => $e->getMessage()
            ], 500);
        }
    }
}
