import { InputAssetProps } from './assets';
import { FontWithFamily } from './shared';
export interface FontAssetProps extends InputAssetProps {
    id: string;
    type: 'font';
    name: string;
    customData: {
        font: FontWithFamily;
        needsLoading: boolean;
    };
}
export interface FontOption {
    id: string;
    label: string;
}
export type DefaultFont = string | FontOption | FontWithFamily;
export interface FontsConfig {
    /**
     * Enable Font Manager.
     * @default false
     */
    enableFontManager?: boolean;
    /**
     * An array of {@type Font} objects to add by default to the loaded project.
     * Instead of a {@type Font} object, you may use a valid {@link https://developer.mozilla.org/en-US/docs/Web/CSS/font-family|css font-family string}.
     * You may pass a function to extend the default fonts.
     * @example
     * default: [
     *   // Add a font with a font file url to the project.
     *   {
     *     family: "Aboreto",
     *     variants: {
     *       regular: {
     *         family: "Aboreto",
     *         source: "https://fonts.gstatic.com/s/aboreto/v2/5DCXAKLhwDDQ4N8blKHeA2yuxSY.woff2",
     *         variant: "regular"
     *       }
     *     }
     *   },
     *   // Add a font without a font file, rely on system fonts with fallbacks.
     *   { id: 'Arial, Helvetica, sans-serif', label: 'Arial' }
     * ]
     * @example
     * // Pass a function to extend the default fonts
     * default: ({ baseDefault }) => ([myCustomFont, ...baseDefault])
     */
    default?: DefaultFont[] | ((props: {
        baseDefault: DefaultFont[];
    }) => DefaultFont[]);
    /**
     * If true, shows a panel to manage project fonts in the global page settings.
     */
    showProjectFonts?: boolean;
    /**
     * If true, shows a button to add fonts to the project.
     * If there are no font providers available, the button is not displayed.
     */
    addFonts?: boolean;
}
