import React, { useRef, useState } from 'react'

import Input from '../Input'
import FileInput from '../FileInput'
import Button from '../../Components/Button';

import TabControl from '../TabControl';
import Tab from '../Tab';
import Table from '../TableLayout/Table';
import TableRow from '../TableLayout/TableRow';
import TableColumn from '../TableLayout/TableColumn';
import ToggleSwitch from '../ToggleSwitch';

import useGetFormattedDate from '../hooks/useGetFormattedDate';
import RichTextEditor from '../RichTextEditor';

import { toast } from 'react-toastify';

import { useSelector } from 'react-redux'
import BlocksForm from './BlocksForm';

import { v4 as uuidv4 } from 'uuid';
import TestimonialsForm from './TestimonialsForm';
import ServicesForm from './ServicesForm';


const
    ContactsForm = ({ onSubmit = (data) => { }, DefaultData, label = 'Add New Contact', social_types, references, restricted }) => {

        const roles = useSelector(state => state.user.roles)

        //state variables
        const [name, setName] = useState(DefaultData?.name ?? null);
        const [code, setCode] = useState(DefaultData?.code ?? null);
        const [number, setNumber] = useState(DefaultData?.number ?? null)
        const [mobile, setMobile] = useState(DefaultData?.mobile ?? null)
        const [description, setDescription] = useState(DefaultData?.description ?? null)
        const [address, setAddress] = useState(DefaultData?.address ?? null)
        const [profile, setProfile] = useState(DefaultData?.profile_pic ?? null);
        const [template, setTemplate] = useState(DefaultData?.template ?? 'template1')
        const [cover, setCover] = useState(DefaultData?.cover_pic ?? null)
        const [published, setPublished] = useState(DefaultData?.published ?? false)
        const [poweredBy, setPoweredBy] = useState(DefaultData?.powered_by ?? 'Twister Technologies')
        const [gender, setGender] = useState(DefaultData?.gender ?? 'Male')
        const [subscriptionDate, setSubscriptionDate] = useState(DefaultData?.subscription_date ?? useGetFormattedDate({}))
        const [subscriptionEndDate, setSubscriptionEndDate] = useState(DefaultData?.subscription_end_date ?? useGetFormattedDate({ add: { year: 1, month: 0, day: 0 } }))

        const [reference1, setReference1] = useState(DefaultData?.reference1 ?? '')
        const [reference2, setReference2] = useState(DefaultData?.reference2 ?? '')

        const [embedsTitle, setEmbedsTitle] = useState(DefaultData?.embeds_section ?? null)
        const [blocksTitle, setBlocksTitle] = useState(DefaultData?.blocks_section ?? null)
        const [testimonialsTitle, setTestimonialsTitle] = useState(DefaultData?.testimonials_section ?? null)
        const [servicesTitle, setServicesTitle] = useState(DefaultData?.services_section ?? null)


        const [loading, setLoading] = useState(false);
        const [codeAvailable, setCodeAvailable] = useState(true);

        const [socialMedia, setSocialMedia] = useState(DefaultData?.social_links ?? []);
        const [videoEmbeds, setVideoEmbeds] = useState(DefaultData?.embeds ?? []);

        const [blocks, setBlocks] = useState(DefaultData?.blocks ?? []);
        const [selectedBlock, setSelectedBlock] = useState(null);

        const [services, setServices] = useState(DefaultData?.services ?? []);
        const [selectedService, setSelectedService] = useState(null);

        const [testimonials, setTestimonials] = useState(DefaultData?.testimonials ?? []);
        const [selectedTestimonial, setSelectedTestimonial] = useState(null);


        //current data
        const [socialType, setSocialType] = useState(1);
        const socialRef = useRef(null);
        const [embedFrame, setEmbedFrame] = useState(null);

        const isFormDisabled = () => {
            if (restricted) {
                return false
            }

            if (DefaultData?.id) {
                return roles?.contactsE == false
            }

            return roles?.contactsA == false
        }

        // Handle adding/updating social media entries
        const handleAddSocialMedia = ({ index, data }) => {
            setSocialMedia((prevMedia) => {
                const newMedia = [...prevMedia];
                newMedia[index] = { ...newMedia[index], ...data };
                return newMedia;
            });
        };

        const handleDeleteSocialMedia = (indexToDelete) => {
            setSocialMedia((prevMedia) => {
                return prevMedia.filter((_, index) => index !== indexToDelete);
            });
        };


        // Handle adding/updating social media entries
        const handleAddVideoEmbeds = ({ index, data }) => {
            setVideoEmbeds((prevEmbeds) => {
                const newEmbeds = [...prevEmbeds];
                newEmbeds[index] = { ...newEmbeds[index], ...data };
                return newEmbeds;
            });
        };

        const handleDeleteVideoEmbeds = (indexToDelete) => {
            setVideoEmbeds((prevEmbeds) => {
                return prevEmbeds.filter((_, index) => index !== indexToDelete);
            });
        };

        const handleBlocksChange = (data) => {

            if(!data?.title){
                toast.error('Title is required for blocks. Please add a title.');
                return;
            }
      
            if (data?.id) {
                setBlocks(prevBlocks => prevBlocks.map(block => block.id === data.id ? { ...block, ...data } : block));
                setSelectedBlock({
                    title: '',
                    description: '',
                    button: '',
                    order: blocks.length > 0 ? parseInt(blocks[blocks.length - 1].order, 10) + 1 : 1,
                    background: '#FFFFFF',
                    link: '',

                });

            } else {
                setBlocks(prevBlocks => [...prevBlocks, { ...data, id: uuidv4(), }])
                setSelectedBlock({
                    title: '',
                    description: '',
                    button: '',
                    order: data?.order ? parseInt(data?.order,10) + 1: 1,
                    background: '#FFFFFF',
                    link: '',

                });
            }


        }


        const handleServicesChange = (data) => {

            if (!data?.title) {
                toast.error('Title is required for services. Please add a title.');
                return;
            }

            if (data?.id) {
                setServices(prevServices => prevServices.map(service => service.id === data.id ? { ...service, ...data } : service));
                setSelectedService({
                    title: '',
                    description: '',
                    button: '',
                    order: service.length > 0 ? parseInt(services[services.length - 1].order, 10) + 1 : 1,
                    background: '#FFFFFF',
                    link: '',

                });

            } else {
                setServices(prevServices => [...prevServices, { ...data, id: uuidv4(), }])
                setSelectedService({
                    title: '',
                    description: '',
                    button: '',
                    order: data?.order ? parseInt(data?.order, 10) + 1 : 1,
                    background: '#FFFFFF',
                    link: '',

                });
            }


        }

        const handleTestimonialsChange = (data) => {

            if (data?.id) {
                setTestimonials(prevTestimonials => prevTestimonials.map(testimonial => testimonial.id === data.id ? { ...testimonial, ...data } : testimonial));
                setSelectedTestimonial({
                    name: '',
                    description: '',
                    order: blocks.length > 0 ? parseInt(testimonials[testimonials.length-1].order, 10) + 1 : 1,
                    image: ''

                });

            } else {
                setTestimonials(prevTestimonials => [...prevTestimonials, { ...data, id: uuidv4(), }])
                setSelectedTestimonial({
                    name: '',
                    description: '',
                    order: data?.order ? parseInt(data?.order, 10) + 1 : 1,
                    image: ''

                });
            }

            

        }

        const handleCancelBlock = () => {
            setSelectedBlock({
                title: '',
                description: '',
                button: '',
                order: blocks.length > 0 ? parseInt(blocks[blocks.length - 1].order, 10) + 1 : 1,
                background: '#FFFFFF',
                link: '',

            });
        }

        const handleCancelService = () => {
            setSelectedService({
                title: '',
                description: '',
                button: '',
                order: services.length > 0 ? parseInt(services[services.length - 1].order, 10) + 1 : 1,
                background: '#FFFFFF',
                link: '',

            });
        }

        const handleCancelTestimonial = () => {
            setSelectedTestimonial(null);
        }

        const handleDeleteBlock = (id) => {

            if (id == null) {
                return
            }
            setBlocks((prevBlocks) => {
                return prevBlocks.filter((item) => item?.id != id);
            });
        };

        const handleDeleteService = (id) => {

            if (id == null) {
                return
            }
            setServices((prevServices) => {
                return prevServices.filter((item) => item?.id != id);
            });
        };

        const handleDeleteTestimonial = (id) => {

            if (id == null) {
                return
            }
            setTestimonials((prevTestimonials) => {
                return prevTestimonials.filter((item) => item?.id != id);
            });
        };




        const handleFormSubmit = (e) => {

            e.preventDefault()
       
            onSubmit({
                name: name,
                code: code,
                number: number,
                mobile: mobile,
                social_links: socialMedia,
                embeds: videoEmbeds,
                description: description,
                address: address,
                profile_pic: profile,
                cover_pic: cover,
                template: template ?? 'template1',
                published: published ?? false,
                powered_by: poweredBy,
                subscription_date: subscriptionDate,
                subscription_end_date: subscriptionEndDate,
                gender: gender,
                services_section: servicesTitle,
                embeds_section: embedsTitle,
                blocks_section: blocksTitle,
                blocks: blocks,
                services: services,
                reference1: reference1,
                reference2: reference2,
                testimonials_section: testimonialsTitle,
                testimonials: testimonials
            });
        }

        const handleCodeCheck = (e) => {
            setLoading(true);
            setCodeAvailable(true);
            fetch(`/api/contacts/check-code?code=${e.target.value}&id=${DefaultData?.id ?? null}`, {
                method: 'get',
                headers: {
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                },
            }).then(response => response.json()).then(data => {
                setLoading(false);
                if (data.status == 200) {
                    toast.error('Code is not available');
                    setCodeAvailable(false);
                } else {
                    setCode(e.target.value)
                    toast.success('Code is available');
                }
            }).catch(error => {
                console.log(error)
            })

        }

        const handlePublishForm = (e) => {

            e.preventDefault();
  
            onSubmit({
                name: name,
                code: code,
                number: number,
                mobile: mobile,
                social_links: socialMedia,
                embeds: videoEmbeds,
                description: description,
                address: address,
                profile_pic: profile,
                cover_pic: cover,
                template: template ?? 'template1',
                published: true,
                powered_by: poweredBy,
                subscription_date: subscriptionDate,
                subscription_end_date: subscriptionEndDate,
                gender: gender,
                blocks: blocks,
                services: services,
                reference1: reference1,
                reference2: reference2,
                services_section: servicesTitle,
                embeds_section: embedsTitle,
                blocks_section: blocksTitle,
                testimonials_section: testimonialsTitle,
                testimonials: testimonials
            });
        }
        return (
            <section className='bg-white rounded-[10px] p-5 mt-10'>
                <h1 class="font-bold text-[22px] mb-5 text-gray-900 text-center md:text-start">{label}</h1>
                <TabControl className={'min-h-[230px]'}>

                    <Tab name="General Info">
                        <section className='flex gap-5 mb-2'>
                            <RichTextEditor disabled={isFormDisabled()} data={{ state: name, setState: setName }} className="w-full mb-2" label={'Name'}></RichTextEditor>
                        </section>

                        <section className='flex gap-5 mb-2'>
                            {!restricted ? <Input disabled={loading || isFormDisabled()} labelClassName={`${codeAvailable ? '' : 'text-red-500'}`} inputClassName={`${codeAvailable ? '' : 'border-red-500 border-2 text-red-500'}`} DefaultValue={code ?? null} onBlur={handleCodeCheck} setState={setCode} placeholder={'Code'} className="w-full mb-2" label={'Code'}></Input> : null}
                            {!restricted ? <Input disabled={isFormDisabled()} type="Select" DefaultValue={poweredBy ?? null} setState={setPoweredBy} placeholder={'Powered By'} className="w-full mb-2" label={'Powered By'}>
                                <option value="Twister Technologies">Twister Technologies</option>
                                <option value="Digital Dimensions">Digital Dimensions</option>
                            </Input> : null}
                        </section>

                        <section className='flex gap-5 mb-2'>
                            <Input disabled={isFormDisabled()} DefaultValue={number ?? null} setState={setNumber} placeholder={'Phone Number'} className="w-full mb-2" label={'Phone Number'}></Input>
                            <Input disabled={isFormDisabled()} DefaultValue={mobile ?? null} setState={setMobile} placeholder={'Mobile Number'} className="w-full mb-2" label={'Mobile Number'}></Input>
                        </section>

                        <section className='flex gap-5 mb-2'>
                            <Input disabled={isFormDisabled()} type="Select" DefaultValue={gender ?? null} setState={setGender} placeholder={'Gender'} className="md:w-1/2 mb-2" label={'Gender'}>
                                <option value="Male">Male</option>
                                <option value="Female">Female</option>
                            </Input>
                            <Input disabled={isFormDisabled()} DefaultValue={template ?? null} setState={setTemplate} placeholder={'Template'} className="md:w-1/2 mb-2" label={'Template'}></Input>

                        </section>

                        <section className='flex gap-5 mb-2'>

                            <RichTextEditor disabled={isFormDisabled()} data={{ state: address, setState: setAddress }} className="w-full mb-2" label={'Address'}></RichTextEditor>
                        </section>



                        <section className='flex gap-5 mb-2'>

                            <RichTextEditor disabled={isFormDisabled()} data={{ state: description, setState: setDescription }} className="w-full mb-2" label={'Description'}></RichTextEditor>
                        </section>
                        {!restricted ? <section className='flex gap-5 mb-2'>
                            <Input DefaultValue={reference1 ?? null} disabled={isFormDisabled()} label={references?.reference1 ?? 'Reference 1'} setState={setReference1} className="w-full mb-2" ></Input>
                            <Input DefaultValue={reference2 ?? null} disabled={isFormDisabled()} label={references?.reference2 ?? 'Reference 2'} setState={setReference2} className="w-full mb-2" ></Input>
                        </section> : null}
                        <section className='flex flex-col md:flex-row gap-2 md:gap-5 mb-2'>
                            <FileInput disabled={isFormDisabled()} isPreview={true} image={profile} className='w-full' setState={setProfile} >Profile Picture</FileInput>
                            <FileInput disabled={isFormDisabled()} isPreview={true} image={cover} className='w-full' setState={setCover} >Cover Picture</FileInput>
                        </section>
                    </Tab>
                    <Tab className='overflow-auto' name="Social Media">
                        <Table headers={["Id", "Type", "Link / Data", "Order", "Actions"]}>
                            {socialMedia.map((social, index) => {
                                return (
                                    <TableRow key={`${index}/${social.type}`}>
                                        <TableColumn className='font-bold'>{index + 1}</TableColumn>
                                        <TableColumn ><Input disabled={isFormDisabled()} onChange={(e) => { handleAddSocialMedia({ index: index, data: { social_type_id: e.target.value } }) }} type="Select">
                                            {social_types.map(type => {
                                                return (
                                                    <option selected={social.social_type_id == type.id} key={type.id} value={type.id}>{type.type}</option>
                                                )
                                            })}
                                        </Input>
                                        </TableColumn>
                                        <TableColumn>
                                            <Input disabled={isFormDisabled()} onChange={(e) => { handleAddSocialMedia({ index: index, data: { link: e.target.value } }) }} DefaultValue={social.link} className="w-full"></Input>
                                        </TableColumn>
                                        <TableColumn>
                                            <Input disabled={isFormDisabled()} type="number" onChange={(e) => { handleAddSocialMedia({ index: index, data: { order: e.target.value } }) }} DefaultValue={social.order} className="w-full"></Input>
                                        </TableColumn>
                                        <TableColumn>
                                            <div onClick={() => {
                                                if (isFormDisabled()) {
                                                    return
                                                }

                                                handleDeleteSocialMedia(index)
                                            }} className='text-[11px] text-red-500 font-bold cursor-pointer'>Delete</div>
                                        </TableColumn>
                                    </TableRow>
                                )
                            })}

                            {/* ADD NEW SOCIAL MEDIA ROW */}
                            <TableRow className="bg-gray-100">
                                <TableColumn className="font-bold">{socialMedia.length + 1}</TableColumn>
                                <TableColumn>
                                    <Input
                                        disabled={isFormDisabled()}
                                        type="Select"
                                        setState={setSocialType}
                                    >

                                        {social_types.map((type) => (
                                            <option key={type.id} value={type.id}>
                                                {type.type}
                                            </option>
                                        ))}
                                    </Input>
                                </TableColumn>
                                <TableColumn>
                                    <Input
                                        disabled={isFormDisabled()}
                                        ref={socialRef}
                                        className="w-full"
                                        placeholder="Enter Link"
                                    />
                                </TableColumn>

                                <TableColumn>
                                    <div
                                        className="text-[11px] text-green-500 font-bold cursor-pointer"

                                        onClick={() => {

                                            if (isFormDisabled()) {
                                                return
                                            }
                                            handleAddSocialMedia({
                                                index: socialMedia.length,
                                                data: {
                                                    link: socialRef.current.value, social_type_id: socialType, order: socialMedia.length > 0 ? parseInt(socialMedia[socialMedia.length - 1]?.order, 10) + 1
                                                        : 1
                                                },
                                            })

                                            socialRef.current.value = ''
                                        }
                                        }
                                    >
                                        Add
                                    </div>
                                </TableColumn>
                            </TableRow>
                        </Table>

                    </Tab>

                    <Tab name="Blogs">
                        <Input disabled={isFormDisabled()} DefaultValue={blocksTitle ?? null} setState={setBlocksTitle} placeholder={'Section Title'} className="w-full md:w-1/3 mb-2" label={'Section Title'}></Input>
                        {blocks?.length > 0 ?
                            <Table headers={["title", "description", "Image" , "Button", "Link", "background", "Order", "Actions"]}>
                                {blocks?.map((block) => {
                                    return (
                                        <TableRow key={block?.id}>
                                            <TableColumn ><strong>{block?.title ? block?.title.replace(/<[^>]*>/g, "").slice(0, 20) : ''}</strong></TableColumn>
                                            <TableColumn >{block?.description ? block?.description.replace(/<[^>]*>/g, "").slice(0, 20) : ''}</TableColumn>
                                            <TableColumn >{block?.image ? <img loading='lazy' className='w-[70px] h-[70px] object-cover' src={`${typeof block?.image === 'object' ? URL.createObjectURL(block?.image) : block?.image}`} alt="Block Image"></img> : ''}</TableColumn>
                                            <TableColumn >{block?.button ? block?.button.replace(/<[^>]*>/g, "").slice(0, 20) : ''}</TableColumn>
                                            <TableColumn >{block?.link ?? ''}</TableColumn>
                                            <TableColumn >{block?.background ? <div style={{ backgroundColor: block?.background ?? '#000000'}} className='w-[30px] h-[20px]'></div> : 'No Color Selected'}</TableColumn>
                                            <TableColumn >{block?.order ?? ''}</TableColumn>
                                            <TableColumn>
                                                <div onClick={() => {
                                                    if (isFormDisabled()) {
                                                        return
                                                    }

                                                    handleDeleteBlock(block?.id ?? null)
                                                }} className='text-[11px] text-red-500 font-bold cursor-pointer'>Delete</div>
                                                <div onClick={() => {
                                                    if (isFormDisabled()) {
                                                        return
                                                    }

                                                    setSelectedBlock(block)
                                                }} className='text-[11px] text-blue-500 font-bold cursor-pointer'>Edit</div>
                                            </TableColumn>
                                        </TableRow>
                                    );
                                })}

                            </Table>
                            : ''}

                        <BlocksForm onCancelBlock={handleCancelBlock} data={selectedBlock} onSubmit={handleBlocksChange} isFormDisabled={isFormDisabled} > </BlocksForm>
                    </Tab>



                    <Tab name="Services">
                        <Input disabled={isFormDisabled()} DefaultValue={servicesTitle ?? null} setState={setServicesTitle} placeholder={'Section Title'} className="w-full md:w-1/3 mb-2" label={'Section Title'}></Input>
                        {services?.length > 0 ?
                            <Table headers={["title", "description", "Image", "Button", "Link", "background", "Order", "Actions"]}>
                                {services?.map((service) => {
                                    return (
                                        <TableRow key={service?.id}>
                                            <TableColumn ><strong>{service?.title ? service?.title.replace(/<[^>]*>/g, "").slice(0, 20) : ''}</strong></TableColumn>
                                            <TableColumn >{service?.description ? service?.description.replace(/<[^>]*>/g, "").slice(0, 20) : ''}</TableColumn>
                                            <TableColumn >{service?.image ? <img loading='lazy' className='w-[70px] h-[70px] object-cover' src={`${typeof service?.image === 'object' ? URL.createObjectURL(service?.image) : service?.image}`} alt="service Image"></img> : ''}</TableColumn>
                                            <TableColumn >{service?.button ? service?.button.replace(/<[^>]*>/g, "").slice(0, 20) : ''}</TableColumn>
                                            <TableColumn >{service?.link ?? ''}</TableColumn>
                                            <TableColumn >{service?.background ? <div style={{ backgroundColor: service?.background ?? '#000000' }} className='w-[30px] h-[20px]'></div> : 'No Color Selected'}</TableColumn>
                                            <TableColumn >{service?.order ?? ''}</TableColumn>
                                            <TableColumn>
                                                <div onClick={() => {
                                                    if (isFormDisabled()) {
                                                        return
                                                    }

                                                    handleDeleteService(service?.id ?? null)
                                                }} className='text-[11px] text-red-500 font-bold cursor-pointer'>Delete</div>
                                                <div onClick={() => {
                                                    if (isFormDisabled()) {
                                                        return
                                                    }

                                                    setSelectedService(service)
                                                }} className='text-[11px] text-blue-500 font-bold cursor-pointer'>Edit</div>
                                            </TableColumn>
                                        </TableRow>
                                    );
                                })}

                            </Table>
                            : ''}

                        <ServicesForm onCancelService={handleCancelService} data={selectedService} onSubmit={handleServicesChange} isFormDisabled={isFormDisabled} > </ServicesForm>
                    </Tab>


                    <Tab name="Testimonials">
                        <Input disabled={isFormDisabled()} DefaultValue={testimonialsTitle ?? null} setState={setTestimonialsTitle} placeholder={'Section Title'} className="w-1/3 mb-2" label={'Section Title'}></Input>
                        {testimonials?.length > 0 ?
                            <Table headers={["Name", "Description", "Image", "Order", "Actions"]}>
                                {testimonials?.map((testimonial) => {
                                    return (
                                        <TableRow key={testimonial?.id}>
                                            <TableColumn ><strong>{testimonial?.name ? testimonial?.name.replace(/<[^>]*>/g, "").slice(0, 20) : ''}</strong></TableColumn>
                                            <TableColumn >{testimonial?.description ? testimonial?.description.replace(/<[^>]*>/g, "").slice(0, 20) : ''}</TableColumn>
                                            <TableColumn >{testimonial?.image ? <img loading='lazy' className='w-[70px] h-[70px] object-cover' src={`${typeof testimonial?.image === 'object' ? URL.createObjectURL(testimonial?.image) : testimonial?.image}`} alt="Block Image"></img> : ''}</TableColumn>
                                            <TableColumn >{testimonial?.order ?? ''}</TableColumn>
                                            <TableColumn>
                                                <div onClick={() => {
                                                    if (isFormDisabled()) {
                                                        return
                                                    }

                                                    handleDeleteTestimonial(testimonial?.id ?? null)
                                                }} className='text-[11px] text-red-500 font-bold cursor-pointer'>Delete</div>
                                                <div onClick={() => {
                                                    if (isFormDisabled()) {
                                                        return
                                                    }

                                                    setSelectedTestimonial(testimonial)
                                                }} className='text-[11px] text-blue-500 font-bold cursor-pointer'>Edit</div>
                                            </TableColumn>
                                        </TableRow>
                                    );
                                })}

                            </Table>
                            : ''}

                        <TestimonialsForm onCancelTestimonial={handleCancelTestimonial} data={selectedTestimonial} onSubmit={handleTestimonialsChange} isFormDisabled={isFormDisabled} > </TestimonialsForm>
                    </Tab>

                    <Tab name="Video Embeds">
                        <Input disabled={isFormDisabled()} DefaultValue={embedsTitle ?? null} setState={setEmbedsTitle} placeholder={'Section Title'} className="w-1/3 mb-2" label={'Section Title'}></Input>

                        <Table headers={["Id", "Embed", "Order", "Actions"]}>
                            {videoEmbeds.map((videoEmbed, index) => {
                                return (
                                    <TableRow key={`${index}/${videoEmbed.order}`}>
                                        <TableColumn className='font-bold'>{index + 1}</TableColumn>
                                        <TableColumn>
                                            <Input disabled={isFormDisabled()} onChange={(e) => { handleAddVideoEmbeds({ index: index, data: { embed: e.target.value } }) }} DefaultValue={videoEmbed.embed} className="w-full"></Input>
                                        </TableColumn>
                                        <TableColumn>
                                            <Input disabled={isFormDisabled()} type="number" onChange={(e) => { handleAddVideoEmbeds({ index: index, data: { order: e.target.value } }) }} DefaultValue={videoEmbed.order} className="w-full"></Input>
                                        </TableColumn>
                                        <TableColumn>
                                            <div onClick={() => {
                                                if (isFormDisabled()) {
                                                    return
                                                }

                                                handleDeleteVideoEmbeds(index)
                                            }} className='text-[11px] text-red-500 font-bold cursor-pointer'>Delete</div>
                                        </TableColumn>
                                    </TableRow>
                                )
                            })}

                            {/* ADD NEW VIDEO EMBED ROW */}
                            <TableRow className="bg-gray-100">
                                <TableColumn className="font-bold">{videoEmbeds?.length + 1}</TableColumn>

                                <TableColumn>
                                    <Input
                                        disabled={isFormDisabled()}
                                        setState={setEmbedFrame}
                                        className="w-full"
                                        placeholder="Video Embed"
                                    />
                                </TableColumn>

                                <TableColumn>
                                    <div
                                        className="text-[11px] text-green-500 font-bold cursor-pointer"

                                        onClick={() => {
                                            if (isFormDisabled()) {
                                                return
                                            }

                                            handleAddVideoEmbeds({
                                                index: videoEmbeds.length,
                                                data: { embed: embedFrame, order: videoEmbeds.length > 0 ? parseInt(videoEmbeds[videoEmbeds.length - 1]?.order, 10) + 1 : 1 },
                                            })
                                        }}
                                    >
                                        Add
                                    </div>
                                </TableColumn>
                            </TableRow>
                        </Table>
                    </Tab>
                    {!restricted && <Tab name="Subscription">
                        <section className='flex gap-5 mb-2'>
                            <Input disabled={isFormDisabled()} label={`Start Date (${subscriptionDate ?? null})`} type="Date" setState={setSubscriptionDate} className="w-full mb-2" ></Input>
                            <Input disabled={isFormDisabled()} label={`End Date (${subscriptionEndDate ?? null})`} type="Date" setState={setSubscriptionEndDate} className="w-full mb-2" ></Input>
                        </section>
                    </Tab>}

                    {!restricted ? <Tab name="Misc / References"> </Tab> : null}
                </TabControl>

                {roles?.contactsP == true ? <ToggleSwitch setState={setPublished} DefaultChecked={published} className="mt-10" >Publish</ToggleSwitch> : ''}
                <section className="mt-2 gap-2 md:gap-5 flex flex-col md:flex-row items-center justify-center">
                    {(published == false && roles?.contactsP == true) ? <Button onClick={handlePublishForm} className='bg-gray-900 hover:bg-[#000000]  mx-auto'>Save And Publish</Button> : ''}
                    <Button disabled={isFormDisabled()} onClick={handleFormSubmit} className=' bg-gray-900 hover:bg-[#000000]  mx-auto'>Save</Button>
                </section>
            </section>
        )
    }

export default ContactsForm