import React, { useState, useEffect, useRef } from 'react'

import Input from '../Input'
import FileInput from '../FileInput'
import Button from '../../Components/Button';
import RichTextEditor from '../RichTextEditor';


const ServicesForm = ({ data, isFormDisabled, onSubmit, onCancelService }) => {
    const [formData, setFormData] = useState(data ?? {
        title: '',
        description: '',
        button: '',
        order: 0,
        background: '#FFFFFF',
        link: '',

    });

    const backgroundRef = useRef(data?.background ?? '#FFFFFF');
    const linkRef = useRef(data?.link ?? '');
    const orderRef = useRef(data?.order ?? '');

    useEffect(() => {
        setFormData(data);
        backgroundRef.current.value = data?.background ?? '#FFFFFF';
        linkRef.current.value = data?.link ?? '';
        orderRef.current.value = data?.order ?? '';
    }, [data])

    const handleSubmit = (e) => {
        e.preventDefault();
        onSubmit({ ...formData, link: linkRef.current.value, background: backgroundRef.current.value, order: orderRef.current.value });
        setFormData({
            title: '',
            description: '',
            button: '',
            order: 0,
            background: '#000000',
            link: '',
        })
    }

    return (
        <form onSubmit={handleSubmit} className='mt-5 '>
            <h3 class="font-bold text-[18px] mb-2 text-gray-900 text-center md:text-start">{data?.title ? `Edit Block (${data?.title.replace(/<[^>]*>/g, "").slice(0, 20)})` : 'Add New Block'}</h3>
            <section className='flex grid grid-cols-2 md:grid-cols-3 gap-x-5 gap-y-2 md:w-max'>
                <RichTextEditor
                    required={true}
                    disabled={isFormDisabled()}
                    data={{
                        state: formData?.title,
                    }}
                    onBlur={(newContent) => setFormData((prevState) => ({ ...prevState, title: newContent }))}
                    className="w-full max-md:col-span-2 md:max-w-[300px]"
                    label={'Title'}
                />
                <RichTextEditor
                    disabled={isFormDisabled()}
                    data={{
                        state: formData?.description,
                    }}
                    onBlur={(newContent) => setFormData((prevState) => ({ ...prevState, description: newContent }))}
                    className="w-full max-md:col-span-2 md:max-w-[300px]"
                    label={'Description'}
                />
                <RichTextEditor
                    disabled={isFormDisabled()}
                    data={{
                        state: formData?.button,
                    }}
                    onBlur={(newContent) => setFormData((prevState) => ({ ...prevState, button: newContent }))}
                    className="w-full max-md:col-span-2 md:max-w-[300px]"
                    label={'Button'}
                />

                <Input
                    ref={orderRef}
                    label="Order"
                    disabled={isFormDisabled()}
                    type="number"
                    defaultValue={data?.order ?? 0}
                    className="w-full"
                />

                <Input
                    ref={backgroundRef}
                    disabled={isFormDisabled()}
                    defaultValue={data?.background ?? '#FFFFFF'}
                    inputClassName="min-h-[42px] bg-white"
                    className="w-full"
                    type="color"
                    label={'Background'}
                />
                <Input
                    ref={linkRef}
                    label="Button Link"
                    disabled={isFormDisabled()}
                    type="text"

                    defaultValue={data?.link ?? ''}
                    className="w-full"
                />

                <FileInput
                    disabled={isFormDisabled()}
                    isPreview={true}
                    image={formData?.image}
                    className='w-full max-md:col-span-2'
                    setState={(image) => setFormData((prevState) => ({ ...prevState, image: image }))}
                >
                    Image
                </FileInput>
            </section>
            <section className='flex gap-5 mt-5'>
                <Button
                    type="submit"
                    className='bg-gray-900 hover:bg-[#000000] max-w-fit'
                >
                    {data?.title ? 'Edit Service' : 'Add Service'}
                </Button>
                {data?.title ? <Button
                    onClick={
                        onCancelService
                    }
                    className='bg-gray-900 hover:bg-[#000000] max-w-fit'
                >
                    Cancel
                </Button> : ''}
            </section>

        </form>
    )
}

export default ServicesForm